#include "DDirectoriosPorDefecto.h"
#include <shlobj.h>

#if _MSC_VER > 1400
    // Macro para obtener el directorio ////////////////////////////////////
    #define DWL_OBTENER_DIRECTORIO(CSLID_V5, FOLDERID_V6, DWLSTR)          \
        BOOL Ret = _ObtenerDirectorioV6(FOLDERID_V6, DWLSTR);              \
        if (Ret == FALSE) Ret = _ObtenerDirectorioV5(CSLID_V5, DWLSTR);    \
        return Ret;                                                        \
    ////////////////////////////////////////////////////////////////////////
#else
    // Macro para obtener el directorio ///////////////////////////
    #define DWL_OBTENER_DIRECTORIO(CSLID_V5, FOLDERID_V6, DWLSTR) \
        return _ObtenerDirectorioV5(CSLID_V5, DWLSTR);            \
    ///////////////////////////////////////////////////////////////
#endif

namespace DWL {

#if _MSC_VER > 1400
    //! Constructor.
    /*! Constructor.
		    \fn			DDirectoriosPorDefecto(void);
		    \return		No devuelve nada.
    */
    DDirectoriosPorDefecto::DDirectoriosPorDefecto(void) : _Shell32(NULL), _VersionMayor(0), _SHGetKnownFolderPath(NULL)  { 
    }
#else
    //! Constructor.
    /*! Constructor.
		    \fn			DDirectoriosPorDefecto(void);
		    \return		No devuelve nada.
    */
    DDirectoriosPorDefecto::DDirectoriosPorDefecto(void) : _Shell32(NULL), _VersionMayor(0), _SHGetFolderPath(NULL) { 
    }
#endif

    //! Destructor.
    /*! Destructor.
	        \fn			~DDirectoriosPorDefecto(void);
	        \return		No devuelve nada.
    */
    DDirectoriosPorDefecto::~DDirectoriosPorDefecto(void) {
	    if (_Shell32 != NULL) FreeLibrary(_Shell32);
    }


    //! Funcin que llamara a SHGetFolderPath si es posible.
    /*! Esta funcin llamara a SHGetFolderPath si es posible.
		    \fn			BOOL _ObtenerDirectorioV5(int CSLID, DString &StrOut);
            \param[in]  CSLID  : ID en formato CSLID del directorio a obtener.
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE en caso de exito, FALSE en caso de error.
		    \remarks    Funcin de uso interno para la DWL.
    */
    BOOL DDirectoriosPorDefecto::_ObtenerDirectorioV5(int CSLID, DString &StrOut) {
	    TCHAR TmpStr[MAX_PATH] = TEXT("");
	    if (SUCCEEDED(SHGetFolderPath(NULL, CSLID, NULL, 0, TmpStr))) {
		    StrOut = TmpStr;
		    return TRUE;
	    }
	    return FALSE;
    }

#if _MSC_VER > 1400
    //! Funcin que llamara a SHGetKnownFolderPath si es posible.
    /*! Esta funcin llamara a SHGetKnownFolderPath si es posible.
		    \fn			BOOL _ObtenerDirectorioV5(int CSLID, DString &StrOut);
            \param[in]  rfid   : ID en formato GUID del directorio a obtener.
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE en caso de exito, FALSE en caso de error.
		    \remarks    Funcin de uso interno para la DWL.
    */
    BOOL DDirectoriosPorDefecto::_ObtenerDirectorioV6(const GUID& rfid, DString &StrOut) {
	    if (_VersionMayor > 5 && _SHGetKnownFolderPath != NULL) { // Vista o superior
		    PWSTR Tmp = NULL;
		    if (S_OK == _SHGetKnownFolderPath(rfid, NULL, 0, &Tmp)) {
			    StrOut = Tmp;
			    CoTaskMemFree(Tmp);
			    return TRUE;
		    }
	    }
	    return FALSE;
    }
#endif

    //! Funcin que inicia esta clase.
    /*! Esta funcin inicia la clase de forma que carga la libreria Shell32 y la funcion SHGetKnownFolderPath.
		    \fn			void _Iniciar(void);
		    \return		No devuelve nada.
		    \remarks    Funcin de uso interno para la DWL.
    */
    void DDirectoriosPorDefecto::_Iniciar(OSVERSIONINFOEX VersionWindows) {
/*        OSVERSIONINFOEX osvi;
	    ZeroMemory(&osvi, sizeof(OSVERSIONINFOEX));
	    osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
	    GetVersionEx((OSVERSIONINFO *)&osvi);*/
		
        _VersionMayor = VersionWindows.dwMajorVersion;

	    _Shell32 = NULL;
	    // Hay que pasar la ruta completa de shell32.dll para evitar la vulnerabilidad de la dll injection
		_Shell32 = LoadLibrary(TEXT("shell32.dll"));
#if _MSC_VER > 1400
		_SHGetKnownFolderPath = NULL;
    #if defined UNICODE // No hay version ansi de SHGetKnownFolderPath xd
    	if (_Shell32 != NULL) _SHGetKnownFolderPath = reinterpret_cast<TpSHGetKnownFolderPath>(GetProcAddress(_Shell32, "SHGetKnownFolderPath"));
	#endif
#else 
		_SHGetFolderPath = NULL;
    #if defined UNICODE
		if (_Shell32 != NULL)	_SHGetFolderPath = reinterpret_cast<TpSHGetFolderPath>(GetProcAddress(_Shell32, "SHGetFolderPathW"));
    #else
		if (_Shell32 != NULL)	_SHGetFolderPath = reinterpret_cast<TpSHGetFolderPath>(GetProcAddress(_Shell32, "SHGetFolderPathA"));
    #endif
#endif
    }



    //! Funcin que retorna el directorio AppData para todos los usuarios del sistema.
    /*! Esta funcin devuelve el directorio AppData para todos los usuarios del sistema en la clase DWLString proporcionada.
		    \fn			BOOL AppData(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Application Data",
					    si queremos obtener la ruta para el usuario actual debemos usar la funcin AppData_UsuarioActual.
    */
    BOOL DDirectoriosPorDefecto::AppData(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_APPDATA, FOLDERID_ProgramData, StrOut);
    }

    //! Funcin que retorna el directorio AppData del usuario actual.
    /*! Esta funcin devuelve el directorio AppData comn del usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL AppData_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Application Data",
					    si queremos obtener la ruta para todos los usuarios debemos usar la funcin AppData.
    */
    BOOL DDirectoriosPorDefecto::AppData_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_APPDATA, FOLDERID_RoamingAppData, StrOut);
    }

    //! Funcin que retorna el directorio usado como escritorio comn.
    /*! Esta funcin devuelve el directorio usado como escritorio comn en la clase DWLString proporcionada.
		    \fn			BOOL Escritorio(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Desktop",
					    si queremos obtener la ruta para el usuario actual debemos usar la funcin Escritorio_UsuarioActual.
    */
    BOOL DDirectoriosPorDefecto::Escritorio(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_DESKTOPDIRECTORY, FOLDERID_PublicDesktop, StrOut);
    }

    //! Funcin que retorna el directorio usado como escritorio del usuario actual.
    /*! Esta funcin devuelve el directorio usado como escritorio del usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL Escritorio_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Desktop",
					    si queremos obtener la ruta para todos los usuarios debemos usar la funcin Escritorio.
    */
    BOOL DDirectoriosPorDefecto::Escritorio_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_DESKTOPDIRECTORY, FOLDERID_Desktop, StrOut);
    }

    //! Funcin que retorna el directorio usado como Men inicio -> Programas.
    /*! Esta funcin devuelve el directorio usado como Men inicio -> Programas en la clase DWLString proporcionada.
		    \fn			BOOL Escritorio_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Programs".
    */
    BOOL DDirectoriosPorDefecto::MenuInicio_Programas(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_PROGRAMS, FOLDERID_CommonPrograms, StrOut);
    }

    //! Funcin que retorna el directorio usado como Men inicio -> Programas del usuario actual.
    /*! Esta funcin devuelve el directorio usado como Men inicio -> Programas del usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL MenuInicio_Programas_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Programs".
    */
    BOOL DDirectoriosPorDefecto::MenuInicio_Programas_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_PROGRAMS , FOLDERID_Programs, StrOut);

	    return _ObtenerDirectorioV5(CSIDL_PROGRAMS, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Documentos.
    /*! Esta funcin devuelve el directorio usado como Mis Documentos en la clase DWLString proporcionada.
		    \fn			BOOL MisDocumentos(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Documentos".
    */
    BOOL DDirectoriosPorDefecto::MisDocumentos(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_DOCUMENTS , FOLDERID_PublicDocuments, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Documentos para el usuario actual.
    /*! Esta funcin devuelve el directorio usado como Mis Documentos para el usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL MisDocumentos_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Documents".
    */
    BOOL DDirectoriosPorDefecto::MisDocumentos_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_MYDOCUMENTS , FOLDERID_Documents, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Imagenes.
    /*! Esta funcin devuelve el directorio usado como Mis Imagenes en la clase DWLString proporcionada.
		    \fn			BOOL MisImagenes(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Documents\My Pictures".
    */
    BOOL DDirectoriosPorDefecto::MisImagenes(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_PICTURES , FOLDERID_PublicPictures, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Imagenes para el usuario actual.
    /*! Esta funcin devuelve el directorio usado como Mis Imagenes para el usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL MisImagenes_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Documents\My Pictures".
    */
    BOOL DDirectoriosPorDefecto::MisImagenes_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_MYPICTURES , FOLDERID_Pictures, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mi Msica.
    /*! Esta funcin devuelve el directorio usado como Mi Msica en la clase DWLString proporcionada.
		    \fn			BOOL MiMusica(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Documents\My Music".
    */
    BOOL DDirectoriosPorDefecto::MiMusica(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_MUSIC, FOLDERID_PublicMusic, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mi Msica para el usuario actual.
    /*! Esta funcin devuelve el directorio usado como Mi Msica para el usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL MiMusica_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Documents\My Music".
    */
    BOOL DDirectoriosPorDefecto::MiMusica_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_MYMUSIC, FOLDERID_Music, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Videos.
    /*! Esta funcin devuelve el directorio usado como Mis Videos en la clase DWLString proporcionada.
		    \fn			BOOL MisVideos(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\All Users\Documents\My Videos".
    */
    BOOL DDirectoriosPorDefecto::MisVideos(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_COMMON_VIDEO, FOLDERID_PublicVideos, StrOut);
    }

    //! Funcin que retorna el directorio usado como Mis Videos para el usuario actual.
    /*! Esta funcin devuelve el directorio usado como Mi Msica para el usuario actual en la clase DWLString proporcionada.
		    \fn			BOOL MisVideos_UsuarioActual(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Documents and Settings\USUARIO ACTUAL\Documents\My Videos".
    */
    BOOL DDirectoriosPorDefecto::MisVideos_UsuarioActual(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_MYVIDEO, FOLDERID_Videos, StrOut);
    }

    //! Funcin que retorna el directorio que contiene Windows.
    /*! Esta funcin devuelve el directorio usado directorio  raz de windows en la clase DWLString proporcionada.
		    \fn			BOOL Windows(DString &StrOut);
		    \param[out] StrOut : Clase DWLString en la que se guardara la ruta del directorio.
		    \return		Devuelve TRUE si la operacin se completo con exito, FALSE en caso contrario.
		    \remarks    Esta funcin devuelve una ruta tipo : "C:\Windows".
    */
    BOOL DDirectoriosPorDefecto::Windows(DString &StrOut) {
        DWL_OBTENER_DIRECTORIO(CSIDL_WINDOWS, FOLDERID_Windows, StrOut);
    }


};